//Knowledgedump.org - Text Converter

#include "text_converter.h"
#include "ui_text_converter.h"


Text_converter::Text_converter(QWidget *parent) :
	QWidget(parent),
	ui(new Ui::Text_converter)
{
	ui->setupUi(this);
	ui->input_text->setTabStopDistance(40);
	ui->output_text->setTabStopDistance(40);
	init();
}


Text_converter::~Text_converter()
{
	delete ui;
}


void Text_converter::on_save_button_clicked()
{
	//List of "words" to replace, separated by "___".
	QStringList replace = ui->replace_line->text().split("___");
	//The strings in the replace list are replaced with strings in with_this
	QStringList with_this = ui->with_this_line->text().split("___");

	//Check, whether number of strings are the same. If not, show error window and return.
	if (replace.size() != with_this.size()) {
		QMessageBox::warning(this, tr("Text Converter"),
			tr("Failed to save: Number of words in the \"Replace\" line doesn't match "
				"number of words in the \"with this\" line.\n"
				"Remember: ___ (3 x \"_\") is used as separator."));
		return;
	}

	//Open dialog box, to enter filename and confirm saving.
	QFileDialog dialog(this);
	dialog.setWindowModality(Qt::WindowModal);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.setNameFilter("*.txt");
	dialog.setDefaultSuffix("txt");
	dialog.selectFile("default");   //Sets default filename.

	if (dialog.exec() != QDialog::Accepted)
		return;

	//Filename is set to first selected file, if multiple were specified.
	QString filename(dialog.selectedFiles().first());
	//Create output text-file with replacement pattern.
	QFile output(filename);
	//Can file be written?
	if (!output.open(QIODevice::WriteOnly | QIODevice::Text)) {
		QMessageBox::warning(this, tr("Text Converter"),
			tr("File creation failed."));
		return;
	}

	//Create output stream and write to file. File contains the two lines, separated by \n.
	QTextStream stream(&output);
	stream << ui->replace_line->text() << "\n" << ui->with_this_line->text();

	return;
}


void Text_converter::on_load_button_clicked()
{
	//Open dialog box to select file for loading and get filename.
	QString filename(QFileDialog::getOpenFileName(this));
	if (filename.isEmpty()) {
		return;
	}
	//File readable?
	QFile input(filename);
	if (!input.open(QIODevice::ReadOnly | QIODevice::Text)) {
		QMessageBox::warning(this, tr("Text Converter"),
			tr("File import failed."));
		return;
	}

	//Open input stream, save to strings and check, whether text file contains valid pattern.
	QTextStream stream(&input);
	//1st line should be content for "Replace" text line.
	QString replace_raw(stream.readLine());
	QStringList replace(replace_raw.split("___"));
	//2nd line is content for "with_this" text line.
	QString with_this_raw = stream.readLine();
	QStringList with_this(with_this_raw.split("___"));
	//Pattern valid?
	if (replace.size() != with_this.size()) {
		QMessageBox::warning(this, tr("Text Converter"),
			tr("Failed to load: Invalid pattern."));
		return;
	}

	//Write pattern to lines.
	ui->replace_line->setText(replace_raw);
	ui->with_this_line->setText(with_this_raw);

	return;
}


void Text_converter::on_paste_button_clicked()
{
	QClipboard *clipboard = QGuiApplication::clipboard();
	ui->input_text->setPlainText(clipboard->text());
}


void Text_converter::on_convert_button_clicked()
{
	//Create lists with strings to replace and check for pattern validity first.
	QStringList replace(ui->replace_line->text().split("___"));
	QStringList with_this(ui->with_this_line->text().split("___"));
	if (replace.size() != with_this.size()) {
		QMessageBox::warning(this, tr("Text Converter"),
			tr("Conversion failed: Number of words in the \"Replace\" line doesn't match "
				"number of words in the \"with this\" line.\n"
				"Remember: ___ (3 x \"_\") is used as separator."));
		return;
	}

	//Get input from text box and convert.
	QString converted(ui->input_text->toPlainText());

	for (int count = 0; count < replace.size(); ++count) {
		converted.replace(replace.at(count), with_this.at(count));
	}

	//Write converted text to output box. Keep char format and font.
	ui->output_text->setPlainText(converted);

	return;
}


void Text_converter::on_copy_button_clicked()
{
	QClipboard *clipboard = QGuiApplication::clipboard();
	clipboard->setText(ui->output_text->toPlainText());
}


void Text_converter::init() {
	//Load default.txt as pattern on program startup, if possible.
	QFile input("default.txt");
	if (!input.open(QIODevice::ReadOnly | QIODevice::Text)) {
		return;
	}

	//Open input stream, save to strings and check, whether text file contains valid pattern.
	QTextStream stream(&input);
	//1st line should be content for "Replace" text line.
	QString replace_raw(stream.readLine());
	QStringList replace(replace_raw.split("___"));
	//2nd line is content for "with_this" text line.
	QString with_this_raw = stream.readLine();
	QStringList with_this(with_this_raw.split("___"));
	//Pattern valid?
	if (replace.size() != with_this.size()) {
		return;
	}

	//Write pattern to lines.
	ui->replace_line->setText(replace_raw);
	ui->with_this_line->setText(with_this_raw);

	return;
}
